/*
 * © 2025 BOLDWIN LTD.  All rights reserved.
 * You may not use this file except in compliance with the applicable license granted to you by BOLDWIN LTD.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * Specific authorizations and restrictions shall be provided for in the License.
 */

package com.boldwin.sdk.demo.java.activities;

import android.content.Intent;
import android.os.Bundle;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.View;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import androidx.appcompat.app.AppCompatActivity;
import androidx.databinding.DataBindingUtil;
import androidx.recyclerview.widget.LinearLayoutManager;
import com.boldwin.sdk.demo.java.R;
import com.boldwin.sdk.demo.java.databinding.ActivityMainBinding;
import com.boldwin.sdk.demo.java.utils.Settings;
import com.boldwin.sdk.demo.java.utils.testcases.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

public class MainActivity extends AppCompatActivity {

    private IntegrationKind integrationKind = null;
    private AdFormat adFormat = null;
    private String searchRequest = "";

    private ActivityMainBinding binding;
    private TestCaseAdapter testCaseAdapter;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        binding = DataBindingUtil.setContentView(this, R.layout.activity_main);

        initSpinners();
        initSearch();
        initList();
    }

    private void initSpinners() {
        List<String> integrationKindItems = new ArrayList<>();
        integrationKindItems.add("All");
        for (IntegrationKind kind : IntegrationKind.values()) {
            integrationKindItems.add(kind.getAdServer());
        }

        ArrayAdapter<String> integrationKindAdapter = new ArrayAdapter<>(
                this,
                android.R.layout.simple_spinner_dropdown_item,
                integrationKindItems
        );

        binding.spinnerIntegrationKind.setAdapter(integrationKindAdapter);
        binding.spinnerIntegrationKind.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int position, long l) {
                integrationKind = (position == 0) ? null : IntegrationKind.values()[position - 1];
                Settings.get().setLastIntegrationKindId(position);
                updateList();
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {
            }
        });

        if (IntegrationKind.values().length < Settings.get().getLastIntegrationKindId() + 1) {
            Settings.get().setLastIntegrationKindId(0);
        }
        binding.spinnerIntegrationKind.setSelection(Settings.get().getLastIntegrationKindId());

        List<String> adFormatItems = new ArrayList<>();
        adFormatItems.add("All");
        for (AdFormat format : AdFormat.values()) {
            adFormatItems.add(format.getDescription());
        }

        ArrayAdapter<String> adFormatAdapter = new ArrayAdapter<>(
                this,
                android.R.layout.simple_spinner_dropdown_item,
                adFormatItems
        );

        binding.spinnerAdType.setAdapter(adFormatAdapter);
        binding.spinnerAdType.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            @Override
            public void onItemSelected(AdapterView<?> adapterView, View view, int position, long l) {
                adFormat = (position == 0) ? null : AdFormat.values()[position - 1];
                Settings.get().setLastAdFormatId(position);
                updateList();
            }

            @Override
            public void onNothingSelected(AdapterView<?> adapterView) {
            }
        });

        binding.spinnerAdType.setSelection(Settings.get().getLastAdFormatId());
    }

    private void initSearch() {
        binding.etSearch.addTextChangedListener(new TextWatcher() {
            @Override
            public void beforeTextChanged(CharSequence charSequence, int i, int i1, int i2) {
            }

            @Override
            public void onTextChanged(CharSequence charSequence, int i, int i1, int i2) {
                searchRequest = charSequence.toString();
                updateList();
            }

            @Override
            public void afterTextChanged(Editable editable) {
            }
        });
    }

    private void initList() {
        testCaseAdapter = new TestCaseAdapter(this::showAd);
        binding.rvAdTypes.setLayoutManager(new LinearLayoutManager(this));
        binding.rvAdTypes.setAdapter(testCaseAdapter);
    }

    private void updateList() {
        List<TestCase> filteredList = TestCaseRepository.getList().stream()
                .filter(testCase -> {
                    if (integrationKind != null && testCase.getIntegrationKind() != integrationKind) {
                        return false;
                    }

                    if (adFormat != null && testCase.getAdFormat() != adFormat) {
                        return false;
                    }

                    if (!searchRequest.isEmpty() && !testCase.getTitle().toLowerCase().contains(searchRequest.toLowerCase())) {
                        return false;
                    }

                    return true;
                })
                .collect(Collectors.toList());

        testCaseAdapter.setList(filteredList);
    }

    private void showAd(TestCase testCase) {
        TestCaseRepository.setLastTestCase(testCase);
        startActivity(new Intent(this, testCase.getActivity()));
    }
}