/*
 * © 2025 BOLDWIN LTD.  All rights reserved.
 * You may not use this file except in compliance with the applicable license granted to you by BOLDWIN LTD.
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * Specific authorizations and restrictions shall be provided for in the License.
 */

package com.boldwin.sdk.demo.kotlin.examples.inapp

import android.os.Bundle
import android.util.Log
import com.boldwin.sdk.api.BoldwinSDK
import com.boldwin.sdk.api.adunits.BoldwinAdView
import com.boldwin.sdk.api.listeners.BoldwinAdViewListener
import com.boldwin.sdk.api.listeners.BoldwinAdViewSizeListener
import com.boldwin.sdk.api.utils.BoldwinAd
import com.boldwin.sdk.api.utils.BoldwinAdSize
import com.boldwin.sdk.api.utils.BoldwinError
import com.boldwin.sdk.api.utils.targeting.BoldwinTargeting
import com.boldwin.sdk.api.utils.targeting.app.BoldwinContent
import com.boldwin.sdk.api.utils.targeting.app.BoldwinContentProducer
import com.boldwin.sdk.api.utils.targeting.app.BoldwinPublisher
import com.boldwin.sdk.api.utils.targeting.app.BoldwinTargetingApp
import com.boldwin.sdk.api.utils.targeting.common.BoldwinTargetingData
import com.boldwin.sdk.api.utils.targeting.common.BoldwinTargetingExt
import com.boldwin.sdk.api.utils.targeting.common.BoldwinTargetingSegment
import com.boldwin.sdk.api.utils.targeting.user.BoldwinExtendedId
import com.boldwin.sdk.api.utils.targeting.user.BoldwinGeo
import com.boldwin.sdk.api.utils.targeting.user.BoldwinTargetingUser
import com.boldwin.sdk.api.utils.targeting.user.BoldwinUniqueId
import com.boldwin.sdk.demo.kotlin.activities.BaseAdActivity


class InAppDisplayBannerWithTargetingActivity : BaseAdActivity() {

    companion object {
        const val PLACEMENT_ID = "686e42a0035b2ced064cc938"
    }

    private var adView: BoldwinAdView? = null

    override var reloadAd = { createAd() }

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        setTargeting()
        createAd()
    }

    private fun createAd() {
        adWrapperView.removeAllViews()

        // 1. Create BoldwinAdView
        val adView = BoldwinAdView(this).apply { adView = this }

        // 2. Configure ad unit
        adView.setPlacementId(PLACEMENT_ID)
        adView.setAdSizes(BoldwinAdSize.BANNER_300x250)
        adView.setAdListener(createListener())
        adView.setSizeListener(createSizeListener())
        adView.setAutoRefreshPeriod(refreshTimeSeconds)

        // 3. Load ad
        adView.loadAd()

        // 4. Add ad view to the app UI
        adWrapperView.addView(adView)
    }

    private fun setTargeting() {
        val targeting = BoldwinTargeting.Builder()
            .app(createApp())
            .user(createUser())
            .build()

        BoldwinSDK.setTargeting(targeting)
    }

    private fun createContent(): BoldwinContent {
        val segmentExt = BoldwinTargetingExt()
        segmentExt.put("ext_key", "Test value for BoldwinTargetingSegment")
        val segment = BoldwinTargetingSegment.Builder()
            .id("segment_001")
            .name("Test Segment")
            .value("test_value")
            .ext(segmentExt)
            .build()


        val dataExt = BoldwinTargetingExt()
        dataExt.put("ext_key", "Test value for BoldwinTargetingData")
        val targetingData = BoldwinTargetingData.Builder()
            .id("data_001")
            .name("Test Data")
            .segment(listOf(segment))
            .ext(dataExt)
            .build()

        val producerExt = BoldwinTargetingExt()
        producerExt.put("ext_key", "Test value for BoldwinContentProducer")
        val producer = BoldwinContentProducer.Builder()
            .id("producer_001")
            .name("Test Producer")
            .cat(listOf("Category1", "Category2"))
            .domain("producer.com")
            .ext(producerExt)
            .build()

        val contentExt = BoldwinTargetingExt()
        contentExt.put("ext_key", "Test value for BoldwinContent")
        val content = BoldwinContent.Builder()
            .id("content_001")
            .title("Test Content")
            .series("Test Series")
            .season("Season 1")
            .episode(1)
            .artist("Test Artist")
            .genre("Test Genre")
            .album("Test Album")
            .isrc("TEST123456789")
            .producer(producer)
            .url("https://test.com/content")
            .productionQuality(1)
            .context(2)
            .contentRating("PG-13")
            .userRating("4.5")
            .qagMediaRating(1)
            .keywords("test,content,sample")
            .liveStream(0)
            .sourceRelationship(1)
            .len(3600)
            .language("en")
            .embeddable(1)
            .categories(listOf("Category1", "Category2", "Category3"))
            .data(listOf(targetingData))
            .ext(contentExt)
            .build()

        return content
    }

    private fun createApp(): BoldwinTargetingApp {
        val content = createContent()

        val publisherExt = BoldwinTargetingExt()
        publisherExt.put("ext_key", "Test value for BoldwinPublisher")
        val publisher = BoldwinPublisher.Builder()
            .id("pub_001")
            .name("Test Publisher")
            .domain("test.com")
            .categories(mutableListOf("News", "Entertainment"))
            .ext(publisherExt)
            .build()

        val appExt = BoldwinTargetingExt()
        appExt.put("ext_key", "Test value for BoldwinTargetingApp")
        val app = BoldwinTargetingApp.Builder()
            .bundle(packageName)
            .version("1.0.0")
            .name("Test App")
            .storeUrl("https://play.google.com/store/apps/test")
            .domain("app.test.com")
            .paid(1)
            .privacyPolicy(1)
            .keywords("app,test,demo")
            .cat(listOf("IAB1", "IAB2", "IAB3"))
            .sectionCat(listOf("Section1", "Section2"))
            .pageCat(listOf("Page1", "Page2"))
            .publisher(publisher)
            .content(content)
            .ext(appExt)
            .build()

        return app
    }

    private fun createUser(): BoldwinTargetingUser {
        val uniqueIds1 = listOf(
            BoldwinUniqueId.Builder()
                .id("unique_id_123")
                .atype(1)
                .build(),
            BoldwinUniqueId.Builder()
                .id("device_id_456")
                .atype(2)
                .build()
        )

        val extendedId = BoldwinExtendedId.Builder()
            .source("adserver.org")
            .uniqueIds(uniqueIds1)
            .build()

        val segmentExt = BoldwinTargetingExt()
        segmentExt.put("ext_key", "Test value for BoldwinTargetingSegment (user)")
        val segment = BoldwinTargetingSegment.Builder()
            .id("segment_001")
            .name("Test Segment")
            .value("test_value")
            .ext(segmentExt)
            .build()

        val geoExt = BoldwinTargetingExt()
        geoExt.put("ext_key", "Test value for BoldwinGeo")
        val geo = BoldwinGeo.Builder()
            .lat(55.7558)
            .lon(37.6173)
            .type(1)
            .accuracy(30)
            .lastfix(300)
            .ipservice(3)
            .country("UKR")
            .region("KYI")
            .regionfips104("RS45")
            .metro("0001")
            .city("Kyiv")
            .zip("101000")
            .utcoffset(180)
            .ext(geoExt)
            .build()

        val dataExt = BoldwinTargetingExt()
        dataExt.put("ext_key", "Test value for BoldwinTargetingData (user)")
        val userData = BoldwinTargetingData.Builder()
            .id("user_001")
            .name("Test User")
            .segment(mutableListOf(segment))
            .ext(dataExt)
            .build()

        val userExt = BoldwinTargetingExt()
        userExt.put("ext_key", "Test value for BoldwinTargetingUser")
        val user = BoldwinTargetingUser.Builder()
            .id("user123")
            .buyerUid("buyer456")
            .yearOfBirth(1990)
            .gender("M")
            .keywords("user_keyword1,user_keyword2")
            .geo(geo)
            .data(mutableListOf(userData))
            .extendedIds(mutableListOf(extendedId))
            .customData("customUserData123")
            .ext(userExt)
            .build()

        return user
    }

    private fun createListener(): BoldwinAdViewListener = object : BoldwinAdViewListener {
        override fun onAdLoaded(bannerView: BoldwinAdView?, ad: BoldwinAd?) {
            Log.d(TAG, "Ad loaded: $ad")
        }

        override fun onAdDisplayed(bannerView: BoldwinAdView?) {
        }

        override fun onAdFailed(
            bannerView: BoldwinAdView?,
            error: BoldwinError?
        ) {
            Log.e(TAG, "Ad failed to load: ${error?.message}")
        }

        override fun onAdClicked(bannerView: BoldwinAdView?) {
        }

        override fun onAdClosed(bannerView: BoldwinAdView?) {
        }
    }

    private fun createSizeListener(): BoldwinAdViewSizeListener {
        return BoldwinAdViewSizeListener { adView, size ->
            Log.d(TAG, "The ad size will change to size: $size")
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        adView?.destroy()
        BoldwinSDK.setTargeting(null)
    }

}
