/*
 © 2025 BOLDWIN LTD.  All rights reserved.
 You may not use this file except in compliance with the applicable license granted to you by BOLDWIN LTD.
 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 Specific authorizations and restrictions shall be provided for in the License.
 */

#import <UIKit/UIKit.h>
#import "TestCasesListViewController.h"
#import "AdFormat.h"
#import "AdFormatDescriptor.h"
#import "IntegrationKind.h"
#import "IntegrationKindDescriptor.h"
#import "TestCase.h"
#import "TestCasesManager.h"

NSString * const cellID = @"TestCaseCell";

@interface TestCasesListViewController ()

@property (weak, nonatomic) IBOutlet UISearchBar *searchBar;
@property (weak, nonatomic) IBOutlet UIButton *integrationKindPicker;
@property (weak, nonatomic) IBOutlet UIButton *adFormatPicker;
@property (weak, nonatomic) IBOutlet UITableView *tableView;

@property (nonatomic) IntegrationKind currentIntegrationKind;
@property (nonatomic) AdFormat currentAdFormat;
@property (nonatomic) NSString *filterText;
@property (nonatomic) NSArray<TestCase *> *displayedCases;

@end

@implementation TestCasesListViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    
    self.tableView.delegate = self;
    self.tableView.dataSource = self;
    self.searchBar.delegate = self;
    
    self.displayedCases = TestCasesManager.allCases;
    
    [self setupPickers];
}

// MARK: - UITableViewDelegate & UITableViewDataSource

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    return self.displayedCases.count;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    
    UITableViewCell *cell = [tableView dequeueReusableCellWithIdentifier:cellID];
    
    UIListContentConfiguration *configuration = cell.defaultContentConfiguration;
    configuration.text = self.displayedCases[indexPath.row].title;
    cell.contentConfiguration = configuration;
    
    return cell;
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    [tableView deselectRowAtIndexPath:indexPath animated:YES];
    
    TestCase * testCase = self.displayedCases[indexPath.row];
    
    UIViewController * viewController = testCase.configurationClosure();
    viewController.view.backgroundColor = [UIColor whiteColor];
    viewController.title = testCase.title;
    
    [self.navigationController pushViewController:viewController animated:YES];
}

// MARK: - UISearchBarDelegate

- (void)searchBar:(UISearchBar *)searchBar textDidChange:(NSString *)searchText {
    self.filterText = searchBar.text ? : @"";
    [self filterTestCases];
}

- (void)searchBarSearchButtonClicked:(UISearchBar *)searchBar {
    self.filterText = searchBar.text ? : @"";
    [self filterTestCases];
    [searchBar endEditing:YES];
}

- (void)filterTestCases {
    self.displayedCases = [[NSMutableArray alloc] init];
    
    self.displayedCases = [TestCasesManager.allCases filteredArrayUsingPredicate:[NSPredicate predicateWithBlock:^BOOL(TestCase * _Nullable evaluatedObject, NSDictionary<NSString *,id> * _Nullable bindings) {
        return self.filterText.length == 0 || [evaluatedObject.title rangeOfString:self.filterText options:NSCaseInsensitiveSearch].length > 0;
    }]];
    
    self.displayedCases = [self.displayedCases filteredArrayUsingPredicate:[NSPredicate predicateWithBlock:^BOOL(TestCase * _Nullable evaluatedObject, NSDictionary<NSString *,id> * _Nullable bindings) {
        return self.currentIntegrationKind == IntegrationKindAll ? true : evaluatedObject.integrationKind == self.currentIntegrationKind;
    }]];
    
    self.displayedCases = [self.displayedCases filteredArrayUsingPredicate:[NSPredicate predicateWithBlock:^BOOL(TestCase * _Nullable evaluatedObject, NSDictionary<NSString *,id> * _Nullable bindings) {
        return self.currentAdFormat == AdFormatAll ? true : [evaluatedObject.adFormats containsObject:@(self.currentAdFormat)];
    }]];
    
    [self.tableView reloadData];
}

- (void)setupPickers {
    UIAction * allIntegrationKindsAction = [UIAction actionWithTitle:@"All" image:nil identifier:nil handler:^(__kindof UIAction * _Nonnull action) {
        self.currentIntegrationKind = IntegrationKindAll;
        [self filterTestCases];
    }];
    
    NSMutableArray<UIAction *> * integrationKindActions = [[NSMutableArray alloc] initWithObjects:allIntegrationKindsAction, nil];
    
    for (int i = 0; i < IntegrationKindAll; ++i) {
        [integrationKindActions addObject:[UIAction actionWithTitle:[IntegrationKindDescriptor getDescriptionForIntegrationKind:(IntegrationKind)i] image:nil identifier:nil handler:^(__kindof UIAction * _Nonnull action) {
            self.currentIntegrationKind = (IntegrationKind)i;
            [self filterTestCases];
        }]];
    }
    
    self.integrationKindPicker.showsMenuAsPrimaryAction = YES;
    self.integrationKindPicker.changesSelectionAsPrimaryAction = YES;
    
    integrationKindActions.firstObject.state = UIMenuElementStateOn;
    
    UIMenu *integrationKindMenu = [UIMenu menuWithChildren:integrationKindActions];
    self.integrationKindPicker.menu = integrationKindMenu;
    
    UIAction * allAdFormatsAction = [UIAction actionWithTitle:@"All" image:nil identifier:nil handler:^(__kindof UIAction * _Nonnull action) {
        self.currentAdFormat = AdFormatAll;
        [self filterTestCases];
    }];
    
    NSMutableArray<UIAction *> * adFormatsActions = [[NSMutableArray alloc] initWithObjects:allAdFormatsAction, nil];
    
    for (int i = 0; i < AdFormatAll; ++i) {
        [adFormatsActions addObject:[UIAction actionWithTitle:[AdFormatDescriptor getDescriptionForAdFormat:(AdFormat)i] image:nil identifier:nil handler:^(__kindof UIAction * _Nonnull action) {
            self.currentAdFormat = (AdFormat)i;
            [self filterTestCases];
        }]];
    }
    
    self.adFormatPicker.showsMenuAsPrimaryAction = YES;
    self.adFormatPicker.changesSelectionAsPrimaryAction = YES;
    
    adFormatsActions.firstObject.state = UIMenuElementStateOn;
    
    UIMenu *adFormatMenu = [UIMenu menuWithChildren:adFormatsActions];
    self.adFormatPicker.menu = adFormatMenu;
    
    self.currentIntegrationKind = IntegrationKindAll;
    self.currentAdFormat = AdFormatAll;
    [self filterTestCases];
}

@end
